{Earth constants. Steven S. Pietrobon 18 Oct 1995.
 Revised 13 July 1998, 6 Jan 2000}

const mu = 3.986005e14; {m^3/s^2, gravitational parameter}
      Re =    6378165.; {m, equatorial radius of planet}
      T  =    86164.09; {s, rotational period}

const bmax = 7; {number of geopotential reference levels}
type tarray = array[0..bmax] of double;
const H_:tarray = (0,11000,20000,32000,47000,51000,71000,84852);
        {geopotential height, m'}
      L_:tarray = (-0.0065,0.0,0.0010,0.0028,0.0,-0.0028,-0.0020,0.0);
        {molecular-scale temperature gradiant, K/m'}
      Rs =     8314.32; {Nm/(kmol K), gas constant}
      M0_ =    28.9644; {kg/kmol, mean molecular weight of air}
      g0 =     9.80665; {m/s^2, acceleration of gravity at 45.5425 deg lat.}
var T_,P_:tarray;
    gs:double;

const n = 15; {number of regions}
type farray = array[0..n] of double;
     carray = array[0..3,0..n] of double;
const Z_:farray = (86000,100000,115000,130000,150000,175000,200000,250000,
                 300000,400000,500000,600000,700000,800000,900000,1000000);
                 {m, geometric altitude}
      Z7 =      86000.; {m, minimum upper atmosphere geometric height}
      Z12 =   1000000.; {m, maximum upper atmosphere geometric height}
var ln_P,ln_rho:carray;

procedure init_atmosphere;
{Initialse atmosphere parameters}

procedure init_lower;
{Initialise lower atmosphere}
const P0 =      101325.; {Pa, sea level air pressure}
      T0 =       288.15; {K, standard sea-level temperature}
var b:integer;
begin{init lower}
  T_[0] := T0;
  P_[0] := P0;
  for b := 0 to bmax-1 do
    begin{layers}
      T_[b+1] := T_[b] + L_[b]*(H_[b+1]-H_[b]);
      if L_[b] = 0.0
        then P_[b+1] := P_[b]*exp(-(g0*M0_/Rs)*(H_[b+1]-H_[b])/T_[b])
        else P_[b+1] := P_[b]*exp((g0*M0_/Rs)*ln(T_[b]/T_[b+1])/L_[b]);
    end;{layers}
end;{init lower}

procedure init_upper;
{Initialise upper atmosphere}
const P:farray = (3.7338e-1,3.2011e-2,4.0096e-3,1.2505e-3,4.5422e-4,1.7936e-4,
                  8.4736e-5,2.4767e-5,8.7704e-6,1.4518e-6,3.0236e-7,8.2130e-8,
                  3.1908e-8,1.7036e-8,1.0873e-8,7.5138e-9);
                 {Pa, pressure}
      rho:farray =(6.958e-6, 5.604e-7, 4.289e-8, 8.152e-9,2.076e-9,6.339e-10,
                  2.541e-10,6.073e-11,1.916e-11,2.803e-12,5.215e-13,1.137e-13,
                  3.070e-14,1.136e-14,5.759e-15,3.561e-15);
                 {kg/m^3, density}
var I:integer;

procedure csi(x:farray;
              var a:carray);
{Cubic spline interpolation}
var h,alpha,l,mu,z:farray;
    I:integer;
begin{csi}
  for I := 0 to n-1 do
    h[I] := x[I+1] - x[I];

  for I := 1 to n-1 do
    alpha[I] := 3*(a[0,I+1]*h[I-1]-a[0,I]*(x[I+1]-x[I-1])+a[0,I-1]*h[I])
                 /(h[I-1]*h[I]);
  l[0] := 1;
  mu[0] := 0;
  z[0] := 0;
  for I := 1 to n-1 do
    begin{step 4}
      l[I] := 2*(x[I+1]-x[I-1]) - h[I-1]*mu[I-1];
      mu[I] := h[I]/l[I];
      z[I] := (alpha[I] - h[I-1]*z[I-1])/l[I];
    end;{step 4}

  l[n] := 1;
  z[n] := 0;
  a[2,n] := z[n];
  for I := n-1 downto 0 do
    begin{step 6}
      a[2,I] := z[I] - mu[I]*a[2,I+1];
      a[1,I] := (a[0,I+1]-a[0,I])/h[I] - h[I]*(a[2,I+1]+2*a[2,I])/3;
      a[3,I] := (a[2,I+1]-a[2,I])/(3*h[I]);
    end;{step 6}
end;{csi}

begin{init upper}
  for I := 0 to n do
    begin{ln}
      ln_P[0,I] := ln(P[I]);
      ln_rho[0,I] := ln(rho[I]);
    end;{ln}
  csi(Z_,ln_P);
  csi(Z_,ln_rho);
end;{init upper}

begin{init atmosphere}
  gs := mu/(Re*Re);
{  writeln('gs = ',gs:7:5,' m/s^2, mu = ',(mu/1e9):8:1,' km^3/s^2. ');}
  init_lower;
  init_upper;
end;{init atmosphere}

procedure atmosphere(Zh:double;
                     var Phr,rho,Cs:double);
{Determine atmosphere pressure, density, and speed of sound.
 Steven S. Pietrobon, 24 Jul 1995. Revised 1 Nov 1999.
 Inputs:  Zh  (m, geometric height above Re)
 Outputs: Phr (pressure relative to surface)
          rho (kg/m^3, density)
          Cs  (m/s, speed of sound)}
const r0 = 6356766.; {m, Earth radius at g0}
      gamma =  1.40; {Ratio of Specific heats for ideal diatomic gas}
var H,Z,P:double;
    b:integer;

procedure lower_atmosphere;
var Tm:double;
begin{lower atmosphere}
  b := 0;
  while H > H_[b+1] do b := b+1;
  Tm := T_[b] + L_[b]*(H-H_[b]);
  if L_[b] = 0.0
    then P := P_[b]*exp(-(g0*M0_/Rs)*(H-H_[b])/T_[b])
    else P := P_[b]*exp((g0*M0_/Rs)*ln(T_[b]/Tm)/L_[b]);
  rho := (P/Tm)*(M0_/Rs);
  Cs := sqrt((gamma*Rs/M0_)*Tm);
end;{lower atmosphere}

procedure upper_atmosphere;
var dZ:double;
begin{upper atmosphere}
  b := 0;
  while Z > Z_[b+1] do b := b+1;
  dZ := Z - Z_[b];
  P := exp(((ln_P[3,b]*dZ+ln_P[2,b])*dZ+ln_P[1,b])*dZ+ln_P[0,b]);
  rho := exp(((ln_rho[3,b]*dZ+ln_rho[2,b])*dZ+ln_rho[1,b])*dZ+ln_rho[0,b]);
  Cs := sqrt((gamma*Rs/M0_)*T_[bmax]);
end;{upper atmosphere}

begin{atmosphere}
  H := (gs/g0)*Re*Zh/(Re+Zh);
  Z := r0*H/(r0-H);
  if Z < Z7
    then lower_atmosphere
    else begin{upper}
           if Z > Z12 then Z := Z12;
           upper_atmosphere;
         end;{upper}
  Phr := P/P_[0];
end;{atmosphere}